#include "Game.h"
#include "Input.h"
#include "SDL2/SDL.h"
#include "Scene.h"
#include <Settings.h>

#define DELTA_TIME_MIN 0.016667f
#define DELTA_TIME_MAX 0.033333f

Game::Game(const char * name) :
    name(name),
    scene(NULL),
    pendingScene(NULL),
    isRunning(true)
{
	renderWindow.Init(name);
    Input::Inst().Reset();
    Input::Inst().ResetKeyboardMapping();
}

bool Game::Tick()
{
    if (pendingScene != NULL)
    {
        if (scene != NULL)
        {
            OnUnloadScene(scene);
        }
        scene = pendingScene;
        pendingScene = NULL;
        scene->state = SceneState_Loaded;
        OnLoadScene(scene);
        scene->Init();
    }

	u32 fps = 60;
	u32 frameTime = 0;
    u32 frameStart = SDL_GetTicks();
    u32 frameDiff = frameStart - frameTick;
    frameTick = frameStart;

    Input & input = Input::Inst();

    renderWindow.Clear();

    float targetDt = 1.0f / (float)(fps > 1 ? fps : 1);
    u32 targetFrameMs = (u32)(targetDt * 1000.0f);
    float dt = (float)frameDiff / 1000.0f;
    if (dt < DELTA_TIME_MIN)
    {
        dt = DELTA_TIME_MIN;
    }
    else if (dt > DELTA_TIME_MAX)
    {
        dt = DELTA_TIME_MAX;
    }

    SDL_Event e;
    while (SDL_PollEvent(&e))
    {
        switch (e.type)
        {
            case SDL_QUIT:
                Stop();
                return false;

            case SDL_WINDOWEVENT:
            {
                if (e.window.event == SDL_WINDOWEVENT_RESIZED) 
                {
                    RenderWindow::Current()->Resize(e.window.data1, e.window.data2);
                }
                break;
            }

            default:
                input.HandleEvent(e);

                if (e.type == SDL_KEYDOWN)
                {
                    if ((Input::IsKeyHeld(SDL_SCANCODE_LALT) || Input::IsKeyHeld(SDL_SCANCODE_RALT)) && Input::IsKeyHeld(SDL_SCANCODE_RETURN))
                    {
                        SettingsData & settings = Settings::Load();
                        settings.windowed = !settings.windowed;
                        Settings::Save();
                        renderWindow.SetFullScreen(!settings.windowed);
                    }
                }

                break;
        }
    }

    input.UpdatePre();
    Update(dt);
    
    if (scene != NULL)
    {
        scene->Update(dt);
    }

    renderWindow.Prepare();

    if (scene != NULL)
    {
        scene->Render();;
    }

    u32 frameMs = SDL_GetTicks() - frameStart;
    i32 renderDiff = (i32)targetFrameMs - (i32)frameMs;
    if (renderDiff > 0)
    {
        SDL_Delay(renderDiff);
    }

    renderWindow.Present();

    input.UpdatePost();
    
    return isRunning;
}

void Game::Stop()
{
    isRunning = false;
}

void Game::SetScene(Scene * value)
{
    if (value == scene)
        return;
    if (scene != NULL)
    {
        scene->state = SceneState_Unloaded;
        scene->OnUnload();
    }
    value->game = this;
    pendingScene = value;
    pendingScene->state = SceneState_Init;
}

Scene * Game::GetScene() 
{ 
    return scene; 
}