#include <Font.h>
#include <read/tinyxml2.h>
#include <Cache.h>
#include <Material.h>
#include <FileSystem.h>

using namespace tinyxml2;
   
static CacheMap<Font> FontCache;

Ref<Font> & Font::Load(const char * pathPrefix)
{
    Ref<Font> & result = FontCache.Get(pathPrefix);
    if (result.IsValid())
        return result;

    Font * font = new Font();
    char texturePath[255];
    sprintf(texturePath, "%s.png", pathPrefix);
    font->material = Material::Load(texturePath, "shaders/ui.dsh");

    char dataPath[32];
    sprintf(dataPath, "%s.fnt", pathPrefix);
    XMLDocument doc;
    FileSystem::LoadXML(FileSystem::GetAbsolutePath(dataPath).c_str(), &doc);
    if (doc.Error())
    {
        Log_Error("Couldn't load font %s", dataPath);
        return NoRef<Font>();
    }

    XMLElement * fontNode = doc.FirstChildElement("font");
    if (fontNode == NULL)
    {
        Log_Error("Couldn't find valid font %s", dataPath);
        return NoRef<Font>();
    }

    XMLElement * commonNode = fontNode->FirstChildElement("common");
    if (commonNode == NULL)
    {
        Log_Error("Couldn't find common for font %s", dataPath);
        return NoRef<Font>();
    }
    font->props.size = commonNode->UnsignedAttribute("base");
    font->props.lineHeight = commonNode->UnsignedAttribute("lineHeight");
    font->props.textureWidth = commonNode->UnsignedAttribute("scaleW");
    font->props.textureHeight = commonNode->UnsignedAttribute("scaleH");

    XMLElement * charsNode = fontNode->FirstChildElement("chars");
    if (charsNode == NULL)
    {
        Log_Error("Couldn't find characters for font %s", dataPath);
        return NoRef<Font>();
    }

    XMLElement * charNode = charsNode->FirstChildElement();
    while (charNode != NULL)
    {
        char c = charNode->UnsignedAttribute("id");
        FontChar * fontChar = &font->chars[c];
        fontChar->x = charNode->UnsignedAttribute("x");
        fontChar->y = charNode->UnsignedAttribute("y");
        fontChar->w = charNode->UnsignedAttribute("width");
        fontChar->h = charNode->UnsignedAttribute("height");
        fontChar->offX = charNode->IntAttribute("xoffset");
        fontChar->offY = charNode->IntAttribute("yoffset");
        fontChar->addX = charNode->IntAttribute("xadvance");
        charNode = charNode->NextSiblingElement();
    }

    FontCache.Set(pathPrefix, font);
    return FontCache.Get(pathPrefix);
}

Font::Font(){}

Font::~Font()
{
    Material::Free(material);
}
    
const FontChar * Font::GetChar(char c) const
{
    return &chars[c];
}