#include <FileSystem.h>
#include <desslibs.h>
#include <string.h>
#include <string>
#include <Mesh.h>
#include <Cache.h>
#ifdef _MSC_BUILD
	#include <read/direntw.h>
#else
	#include <dirent.h>
#endif
#include <Hash.h>
#include <read/tinyxml2.h>

#ifdef _WIN32 || _WIN64
const char PATH_SEPERATOR = '\\';
#else
const char PATH_SEPERATOR = '/';
#endif

std::string FileSystem::GetBaseName(const char * path, bool includeExtension)
{
	char * result = new char[32];
 	char c;
    u8 headI = 0;
    for (u8 i=0; i<UINT8_MAX; i++)
    {
        c = path[i];
        if (c == '/' || c == '\\')
        {
            headI = 0;
        }
        else if ((includeExtension == false && c == '.') || c == '\0')
        {
            result[headI] = '\0';
            break;
        }
        else
        {
            result[headI] = c;
            headI++;
        }
    }
    result[headI] = '\0';
	return result;
}

std::string FileSystem::GetExtension(const char * path)
{
	char * result = new char[8];
 	char c;
    u8 headI = 0;
    for (u8 i=0; i<UINT8_MAX; i++)
    {
        c = path[i];
        if (c == '/' || c == '\\' || c == '.')
        {
            headI = 0;
        }
        else if (c == '\0')
        {
            result[headI] = '\0';
            break;
        }
        else
        {
            result[headI] = c;
            headI++;
        }
	}
	return result;
}

bool FileSystem::IsExtenstion(const char * path, const char * ext)
{
	return HashEqual(GetExtension(path).c_str(), ext);
}

std::string FileSystem::SanitisePath(const char * a)
{
	std::string result = a;
	for (u32 i = 0; i < UINT32_MAX; i++)
	{
		if (result[i] == '\0')
			break;

		if (result[i] == '\\' || result[i] == '/')
		{
			result[i] = PATH_SEPERATOR;
		}
	}
	return result;
}

std::string FileSystem::CombinePath(const char * a, const char * b)
{
	std::string result = SanitisePath(a);
	if (result.back() != PATH_SEPERATOR)
	{
		result += PATH_SEPERATOR;
	}
	result += SanitisePath(b);
	return result;
}

bool FileSystem::IsAbsolutePath(const char * path)
{
	if (path == NULL || path[0] == '\0')
		return false;
	return path[0] == '/' || path[0] == '\\' || path[1] == ':';
}

std::string FileSystem::GetAbsolutePath(const char * path)
{
	if (IsAbsolutePath(path) == false)
	{
		return CombinePath(SDL_GetBasePath(), path);
	}
	return path;
}

std::string FileSystem::GetPrefPath(const char * path)
{
	return CombinePath(SDL_GetPrefPath("Cooked Games", "Sword"), path);
}

std::string FileSystem::LoadText(const char * path)
{
    SDL_RWops * rw = FileOpen(path, "r+b");
	if (rw == NULL) 
	{
		Log_Error("File cannot be read: %s\n", path);
		return "";
	}
	uint64_t fileSize  = SDL_RWsize(rw);
	uint64_t bufferTotal = 0, bufferRead = 1;
	char * output = (char *)malloc(fileSize + 1);
	char * buffer = output;
    while (bufferTotal < fileSize && bufferRead != 0) {
        bufferRead = FileRead(rw, buffer, 1, (fileSize - bufferTotal));
        bufferTotal += bufferRead;
        buffer += bufferRead;
    }
    FileClose(rw);
    if (bufferTotal != fileSize) 
    {
		Log_Error("File size error: %s\n", path);
		free(output);
        return 0;
    }
	output[bufferTotal] = '\0';
	std::string result = output;
	free(output);
    return result;
}

void FileSystem::LoadXML(const char * path, tinyxml2::XMLDocument * doc)
{
	std::string text = LoadText(path);
	if (text.length() > 0)
	{
		doc->Parse(text.c_str());
	}
}

SDL_RWops * FileSystem::FileOpen(const char * path, const char* mode)
{
	std::string fullPath = GetAbsolutePath(path);
	return SDL_RWFromFile(fullPath.c_str(), mode);
}

size_t FileSystem::FileRead(SDL_RWops * file, void * ptr, size_t size, size_t maxnum)
{
	size_t count = SDL_RWread(file, ptr, size, maxnum);
	if (count == 0)
	{
		SDL_RWclose(file);
		return 0;
	}
	return count;
}

size_t FileSystem::FileWrite(SDL_RWops* file, void* ptr, size_t size, size_t maxnum)
{
	size_t count = SDL_RWwrite(file, ptr, size, maxnum);
	if (count == 0)
	{
		SDL_RWclose(file);
		return 0;
	}
	return count;
}

void FileSystem::FileClose(SDL_RWops * file)
{
	SDL_RWclose(file);
}

DirScan FileSystem::DirOpen(const char * path)
{
	DirScan result;
	std::string fullPath = GetAbsolutePath(path);
	result.dir = opendir(fullPath.c_str());
	return result;
}

const char * FileSystem::DirRead(DirScan & dir)
{
	dirent * ent;
	while (ent = readdir((DIR*)dir.dir))
	{
		if (ent->d_name[0] == '.')
			continue;

		return ent->d_name;
	}
	return NULL;
}