#include <Collider.h>
#include <Entity.h>
#include <Maths.h>

bool AABB::Overlaps(const AABB & other) const
{
    if (other.pos.x + other.half.x < pos.x - half.x)
        return false;
        
    if (pos.x + half.x < other.pos.x - other.half.x)
        return false;
    
    if (other.pos.z + other.half.z < pos.z - half.z)
        return false;
        
    if (pos.z + half.z < other.pos.z - other.half.z)
        return false;

    if (other.pos.y + other.half.y < pos.y - half.y)
        return false;

    if (pos.y + half.y < other.pos.y - other.half.y)
        return false;

    return true;
}

bool AABB::Overlaps(Vector3 p) const
{
    if (p.x < pos.x - half.x)
        return false;
        
    if (pos.x + half.x < p.x)
        return false;
    
    if (p.z < pos.z - half.z)
        return false;
        
    if (pos.z + half.z < p.z)
        return false;

    if (p.y < pos.y - half.y)
        return false;

    if (pos.y + half.y < p.y)
        return false;

    return true;
}

void AABB::Resolve(const AABB & other)
{
    float dx = other.pos.x - pos.x;
    float adx = dx > 0.0f ? dx : -dx;
    float dz = other.pos.z - pos.z;
    float adz = dz > 0.0f ? dz : -dz;
    float dy = other.pos.y - pos.y;
    float ady = dy > 0.0f ? dy : -dy;
    float overlapX = adx - half.x - other.half.x;
    float overlapZ = adz - half.z - other.half.z;

    if (other.pos.y + other.half.y < pos.y)
    {
        pos.y = other.pos.y + other.half.y + half.y;
    }
    else if (overlapX > overlapZ)
    {
        if (adx < other.half.x + half.x)
        {
            pos.x = other.pos.x + (dx >= 0.0f ? -1.0f : 1.0f) * (other.half.x + half.x);
        }
    }
    else
    {
        if (adz < other.half.z + half.z)
        {
            pos.z = other.pos.z + (dz >= 0.0f ? -1.0f : 1.0f) * (other.half.z + half.z);
        }
    }
}

const AABB & Collider::GetAABB()
{
    Vector3 half = size * 0.5f;
    if (Maths::Abs(Maths::Dot(owner->transform.Right(), Vector3_RIGHT)) < 0.2f)
    {
        float halfX = half.x;
        half.x = half.z;
        half.z = halfX;
    }
    half.x *= abs(owner->transform.scale.x);
    half.y *= abs(owner->transform.scale.y);
    half.z *= abs(owner->transform.scale.z);
    aabb.pos = owner->transform.position + center;
    aabb.half = half;
    return aabb;
}

Collider::Collider(Entity * _owner, Vector3 _size, Vector3 _center) :
    owner(_owner),
    size(_size),
    center(_center)
{}

bool Collider::CollidesWith(Collider * other)
{
    if (other == NULL)
        return false;

    GetAABB();
    const AABB & b = other->GetAABB();

    return aabb.Overlaps(b);
}

bool Collider::Overlaps(Vector3 p)
{
    GetAABB();
    return aabb.Overlaps(p);
}

void Collider::Resolve(Collider * other)
{
    if (mass > other->mass)
        return;

    if (other == NULL)
        return;

    const AABB & b = other->GetAABB();
    aabb.Resolve(b);

    owner->transform.position = aabb.pos - center;
}